﻿<# 
  Title    : NTLM → Kerberos Toolkit
  Author   : chader.fr
  Created  : 2025-08-31 (UTC)
  License  : MIT
  Note     : Ce script est "signé" par insertion de l'auteur dans l'en-tête.
             (Signature cryptographique non appliquée depuis cet environnement.)
#>

[CmdletBinding()]
param(
  [string[]]$Computers,
  [switch]$AllDomainControllers,
  [datetime]$From = (Get-Date).AddDays(-14),
  [datetime]$To   = (Get-Date),
  [string]$OutputPath = ".\NTLM-Audit-Report",
  [switch]$IncludeBlocked # inclut EventID 4003 (bloqués)
)

$ErrorActionPreference = 'Stop'
$logName = "Microsoft-Windows-NTLM/Operational"
$ids = @(4002)
if ($IncludeBlocked) { $ids += 4003 }

# Résultats en mémoire
$results = New-Object System.Collections.Generic.List[object]

# Résolution des cibles
if ($AllDomainControllers) {
  try {
    Import-Module ActiveDirectory -ErrorAction Stop | Out-Null
    $dcs = (Get-ADDomainController -Filter *).HostName
    $Computers = $dcs
  } catch {
    Write-Warning "Module ActiveDirectory indisponible. Utilisation de la machine locale."
    $Computers = @($env:COMPUTERNAME)
  }
} elseif (-not $Computers) {
  $Computers = @($env:COMPUTERNAME)
}

Write-Host "Cibles: $($Computers -join ', ')"

# Interroge chaque hôte
foreach ($c in $Computers) {
  Write-Host "Lecture des événements NTLM sur $c ..."
  try {
    $q = @{
      LogName   = $logName
      ComputerName = $c
      ErrorAction  = 'Stop'
      FilterHashtable = @{
        Id      = $ids
        StartTime = $From
        EndTime   = $To
      }
    }
    $events = Get-WinEvent @q
  } catch {
    Write-Warning "Impossible de lire $logName sur $c : $_"
    continue
  }

  foreach ($e in $events) {
    # La structure des propriétés varie selon l'ID. On tente une carto prudente.
    $src = $null; $tgt = $null; $ws = $null; $acc = $null
    if ($e.Properties.Count -ge 3) {
      $src = $e.Properties[1].Value
      $tgt = $e.Properties[2].Value
    }
    if ($e.Properties.Count -ge 5) {
      $ws  = $e.Properties[4].Value
    }
    if ($e.Properties.Count -ge 1) {
      $acc = $e.Properties[0].Value
    }
    $results.Add([pscustomobject]@{
      TimeCreated = $e.TimeCreated
      Computer    = $c
      EventId     = $e.Id
      Account     = $acc
      Source      = $src
      Target      = $tgt
      Workstation = $ws
      Message     = $e.Message
    })
  }
}

# Sortie
$new = New-Item -ItemType Directory -Path $OutputPath -Force
$csv = Join-Path $new.FullName ("Audit-NTLM_" + (Get-Date -Format "yyyyMMdd-HHmm") + ".csv")
$results | Sort-Object TimeCreated | Export-Csv -NoTypeInformation -Encoding UTF8 $csv

Write-Host "Terminé. Fichier : $csv"
