﻿<# 
.SYNOPSIS
  Mini-GUI PowerShell WinForms pour RESTAURATION Active Directory guidée.
  - Étape 1 (hors DSRM) : sélection de la version, choix Non/Autoritative, préparation & reboot en DSRM.
  - Étape 2 (en DSRM) : lancement de wbadmin systemstaterecovery, puis (optionnel) ntdsutil authoritative restore.
  - Étape 3 : sortie du DSRM et reboot normal.

.NOTE
  - Exécuter en tant qu'Administrateur.
  - Testé sur Windows Server 2019/2022 avec Windows PowerShell 5.x.
  - Le script persiste l'état dans C:\ProgramData\ADRestore\state.json entre les redémarrages.
#>

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing
$ErrorActionPreference = 'Stop'

# --- Utilitaires ---
$global:StatePath = 'C:\ProgramData\ADRestore'
$global:StateFile = Join-Path $global:StatePath 'state.json'

function Write-Log {
    param([string]$msg)
    $ts = (Get-Date).ToString('yyyy-MM-dd HH:mm:ss')
    $line = "[$ts] $msg"
    $txtLog.AppendText("$line`r`n")
}

function Ensure-StateDir {
    if (-not (Test-Path $global:StatePath)) { New-Item -ItemType Directory -Path $global:StatePath -Force | Out-Null }
}

function Save-State {
    Ensure-StateDir
    $obj = [ordered]@{
        Version = $cmbVersions.SelectedItem
        Mode    = if ($rbAuth.Checked) {'Authoritative'} else {'NonAuthoritative'}
        OUs     = $txtOU.Text
        Stage   = 'PendingRestore'
        When    = (Get-Date).ToString('s')
    }
    $json = $obj | ConvertTo-Json -Depth 4
    Set-Content -Path $global:StateFile -Value $json -Encoding UTF8
    Write-Log "État sauvegardé dans $($global:StateFile)."
}

function Load-State {
    if (Test-Path $global:StateFile) {
        try {
            return Get-Content $global:StateFile -Raw | ConvertFrom-Json
        } catch { return $null }
    }
    return $null
}

function Clear-State {
    if (Test-Path $global:StateFile) { Remove-Item $global:StateFile -Force -ErrorAction SilentlyContinue }
    Write-Log "État nettoyé."
}

function Test-InDSRM {
    # Heuristique : safeboot dsrepair actif ?
    try {
        $out = (bcdedit /enum) 2>$null
        if ($out -match 'safeboot\s+dsrepair') { return $true }
    } catch {}
    # Fallback : service NTDS arrêté en DSRM
    try {
        $svc = Get-Service -Name NTDS -ErrorAction SilentlyContinue
        if ($svc -and $svc.Status -eq 'Stopped') { return $true }
    } catch {}
    return $false
}

function Set-DSRM {
    param([switch]$Enable)
    if ($Enable) {
        Write-Log "Activation du démarrage en DSRM..."
        cmd /c 'bcdedit /set safeboot dsrepair' | Out-Null
    } else {
        Write-Log "Désactivation du démarrage en DSRM..."
        cmd /c 'bcdedit /deletevalue safeboot' | Out-Null
    }
}

function Get-WBVersions {
    $list = New-Object System.Collections.ArrayList
    try {
        $out = & wbadmin get versions
        foreach ($line in $out) {
            if ($line -match 'Version identifier:\s*(.+)$') {
                [void]$list.Add($Matches[1].Trim())
            }
        }
    } catch {
        Write-Log "Erreur lors de la lecture des versions wbadmin : $_"
    }
    return $list
}

function Run-WBRestore {
    param(
        [Parameter(Mandatory=$true)][string]$VersionString
    )
    Write-Log "Lancement de la restauration System State : $VersionString"
    $cmd = "wbadmin start systemstaterecovery -version:`"$VersionString`" -quiet"
    Write-Log "Commande : $cmd"
    $p = Start-Process -FilePath cmd.exe -ArgumentList "/c $cmd" -Wait -PassThru
    if ($p.ExitCode -ne 0) {
        throw "wbadmin a retourné le code $($p.ExitCode). Voir Journaux Microsoft-Windows-Backup."
    }
    Write-Log "Restauration System State terminée."
}

function Run-AuthoritativeRestore {
    param([string[]]$OUs)
    if (-not $OUs -or $OUs.Count -eq 0) {
        Write-Log "Aucun DN d'OU fourni pour la restauration autoritative. Étape ignorée."
        return
    }
    foreach ($dn in $OUs) {
        $dn = $dn.Trim()
        if ([string]::IsNullOrWhiteSpace($dn)) { continue }
        $args = @(
            'activate instance ntds',
            'authoritative restore',
            "restore subtree `"$dn`"",
            'quit','quit'
        )
        Write-Log "NTDSUTIL authoritative restore sur : $dn"
        $p = Start-Process -FilePath ntdsutil.exe -ArgumentList $args -Wait -PassThru
        if ($p.ExitCode -ne 0) {
            throw "ntdsutil a retourné le code $($p.ExitCode) pour $dn"
        }
    }
    Write-Log "Restauration autoritative terminée."
}

# --- UI ---
$form               = New-Object System.Windows.Forms.Form
$form.Text          = "AD Restore Guided (WinForms)"
$form.Size          = New-Object System.Drawing.Size(740, 660)
$form.StartPosition = "CenterScreen"

$lbl1 = New-Object System.Windows.Forms.Label
$lbl1.Text = "Versions disponibles (wbadmin) :"
$lbl1.Location = '20,20'
$lbl1.AutoSize = $true

$cmbVersions = New-Object System.Windows.Forms.ComboBox
$cmbVersions.Location = '20,45'
$cmbVersions.Width = 480
$cmbVersions.DropDownStyle = "DropDownList"

$btnRefresh = New-Object System.Windows.Forms.Button
$btnRefresh.Text = "Rafraîchir"
$btnRefresh.Location = '520,43'
$btnRefresh.Width = 180

$grpMode = New-Object System.Windows.Forms.GroupBox
$grpMode.Text = "Mode de restauration"
$grpMode.Location = '20,90'
$grpMode.Size = New-Object System.Drawing.Size(680,100)

$rbNonAuth = New-Object System.Windows.Forms.RadioButton
$rbNonAuth.Text = "Non autoritative (classique)"
$rbNonAuth.Location = '20,30'
$rbNonAuth.AutoSize = $true
$rbNonAuth.Checked = $true

$rbAuth = New-Object System.Windows.Forms.RadioButton
$rbAuth.Text = "Autoritative (OU/objets)"
$rbAuth.Location = '260,30'
$rbAuth.AutoSize = $true

$lblOU = New-Object System.Windows.Forms.Label
$lblOU.Text = "DN d'OU / objets à restaurer (un par ligne)"
$lblOU.Location = '20,60'
$lblOU.AutoSize = $true

$txtOU = New-Object System.Windows.Forms.TextBox
$txtOU.Location = '300,58'
$txtOU.Width = 360
$txtOU.Height = 40
$txtOU.Multiline = $true
$txtOU.ScrollBars = 'Vertical'

$grpMode.Controls.AddRange(@($rbNonAuth,$rbAuth,$lblOU,$txtOU))

$grpDSRM = New-Object System.Windows.Forms.GroupBox
$grpDSRM.Text = "DSRM (Directory Services Restore Mode)"
$grpDSRM.Location = '20,200'
$grpDSRM.Size = New-Object System.Drawing.Size(680,110)

$lblDS = New-Object System.Windows.Forms.Label
$lblDS.AutoSize = $true
$lblDS.Location = '20,30'

$btnEnter = New-Object System.Windows.Forms.Button
$btnEnter.Text = "Préparer DSRM + redémarrer"
$btnEnter.Location = '20,60'
$btnEnter.Width = 220

$btnExit = New-Object System.Windows.Forms.Button
$btnExit.Text = "Sortir du DSRM + redémarrer"
$btnExit.Location = '260,60'
$btnExit.Width = 220

$grpDSRM.Controls.AddRange(@($lblDS,$btnEnter,$btnExit))

$btnRun = New-Object System.Windows.Forms.Button
$btnRun.Text = "Exécuter la restauration"
$btnRun.Location = '20,325'
$btnRun.Width = 680
$btnRun.Height = 40

$txtLog = New-Object System.Windows.Forms.TextBox
$txtLog.Location = '20,380'
$txtLog.Width = 680
$txtLog.Height = 200
$txtLog.Multiline = $true
$txtLog.ScrollBars = 'Vertical'
$txtLog.ReadOnly = $true
$txtLog.Font = New-Object System.Drawing.Font("Consolas",9)

$form.Controls.AddRange(@($lbl1,$cmbVersions,$btnRefresh,$grpMode,$grpDSRM,$btnRun,$txtLog))

# --- Events ---
$btnRefresh.Add_Click({
    $cmbVersions.Items.Clear() | Out-Null
    $vers = Get-WBVersions
    if ($vers.Count -eq 0) {
        [System.Windows.Forms.MessageBox]::Show("Aucune version wbadmin détectée.","Info",[System.Windows.Forms.MessageBoxButtons]::OK,[System.Windows.Forms.MessageBoxIcon]::Information) | Out-Null
    } else {
        $vers | ForEach-Object { [void]$cmbVersions.Items.Add($_) }
        $cmbVersions.SelectedIndex = 0
        Write-Log "Versions chargées : $($vers.Count)"
    }
})

$btnEnter.Add_Click({
    if (-not $cmbVersions.SelectedItem) {
        [System.Windows.Forms.MessageBox]::Show("Sélectionne une version wbadmin d'abord.","Info") | Out-Null
        return
    }
    Save-State
    try {
        Set-DSRM -Enable
        Write-Log "Redémarrage immédiat en DSRM..."
        Restart-Computer -Force
    } catch {
        Write-Log "Erreur : $_"
        [System.Windows.Forms.MessageBox]::Show("Impossible de redémarrer automatiquement. Redémarre manuellement.","Erreur") | Out-Null
    }
})

$btnExit.Add_Click({
    try {
        Set-DSRM
        Write-Log "Redémarrage immédiat en mode normal..."
        Restart-Computer -Force
    } catch {
        Write-Log "Erreur : $_"
        [System.Windows.Forms.MessageBox]::Show("Impossible de redémarrer automatiquement. Supprime manuellement la valeur safeboot.","Erreur") | Out-Null
    }
})

$btnRun.Add_Click({
    try {
        $inDS = Test-InDSRM
        $state = Load-State
        if ($inDS) {
            if (-not $state -or $state.Stage -ne 'PendingRestore') {
                [System.Windows.Forms.MessageBox]::Show("Aucun état sauvegardé. Sélectionne une version et prépare DSRM d'abord (bouton ci-dessus).","Info") | Out-Null
                return
            }
            Write-Log "En DSRM. État chargé : Version=$($state.Version), Mode=$($state.Mode)"
            Run-WBRestore -VersionString $state.Version
            if ($state.Mode -eq 'Authoritative') {
                $ous = @()
                if ($state.OUs) { $ous = $state.OUs -split "`r?`n" }
                Run-AuthoritativeRestore -OUs $ous
            }
            # Sortie DSRM et reboot
            Set-DSRM
            Clear-State
            Write-Log "Restauration terminée. Redémarrage en mode normal..."
            Restart-Computer -Force
        } else {
            if (-not $cmbVersions.SelectedItem) {
                [System.Windows.Forms.MessageBox]::Show("Sélectionne une version wbadmin.","Info") | Out-Null
                return
            }
            # Préparation DSRM
            Save-State
            [System.Windows.Forms.MessageBox]::Show("L'hôte va redémarrer en DSRM. Relance ce script après connexion en DSRM pour exécuter la restauration.","Information") | Out-Null
        }
    } catch {
        Write-Log "Erreur : $_"
        [System.Windows.Forms.MessageBox]::Show("Erreur: $($_.Exception.Message)","Erreur") | Out-Null
    }
})

$form.Add_Shown({
    $ds = if (Test-InDSRM) { "OUI (DSRM détecté)" } else { "NON (mode normal)" }
    $lblDS.Text = "État actuel : $ds"
})

[void]$form.ShowDialog()
