<# 
.SYNOPSIS
  Sauvegarde System State d’un Contrôleur de Domaine avec rotation et logs.

.PARAMETER BackupTarget
  Cible de sauvegarde (ex: "E:" ou "\\serveur\share").

.PARAMETER KeepVersions
  Nombre de versions System State à conserver (volumes locaux uniquement).

.PARAMETER LogPath
  Chemin du fichier log.

.PARAMETER UseVssCopy
  Utiliser VSS Copy au lieu de VSS Full (par défaut : VSS Full).
#>

[CmdletBinding()]
param(
  [Parameter(Mandatory=$true)]
  [string]$BackupTarget,
  [int]$KeepVersions = 14,
  [string]$LogPath = "C:\Logs\AD-SystemState-Backup.log",
  [switch]$UseVssCopy
)

function Write-Log {
  param([string]$Message, [string]$Level = "INFO")
  $ts = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
  $line = "[$ts][$Level] $Message"
  Write-Output $line
  try {
    $dir = Split-Path -Path $LogPath -Parent
    if (-not (Test-Path $dir)) { New-Item -ItemType Directory -Path $dir -Force | Out-Null }
    Add-Content -Path $LogPath -Value $line
  } catch { Write-Warning "Impossible d'écrire dans le log : $_" }
}

# Vérifier élévation
$principal = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
if (-not $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
  Write-Log "Script non lancé en Administrateur." "ERROR"
  throw "Exécuter ce script en tant qu'Administrateur."
}

Write-Log "=== Début sauvegarde System State ==="
Write-Log "Cible: $BackupTarget | Conserver: $KeepVersions versions | VSS: $([string]::IsNullOrEmpty($UseVssCopy) -replace 'True','Full' -replace 'False','Full')" 

# Installer Windows Server Backup si nécessaire
try {
  $feature = Get-WindowsFeature -Name Windows-Server-Backup -ErrorAction Stop
  if (-not $feature.Installed) {
    Write-Log "Installation de la fonctionnalité Windows-Server-Backup..."
    Install-WindowsFeature -Name Windows-Server-Backup -IncludeManagementTools -ErrorAction Stop | Out-Null
    Write-Log "Windows-Server-Backup installé."
  } else {
    Write-Log "Windows-Server-Backup déjà installé."
  }
} catch {
  Write-Log "Impossible de vérifier/installer Windows-Server-Backup : $_" "ERROR"
  throw
}

# Vérifier la cible
$targetIsLocal = $false
if ($BackupTarget -match '^[A-Za-z]:\\?$') {
  $drive = ($BackupTarget.TrimEnd('\'))
  if (-not (Test-Path "$drive\")) { 
    Write-Log "Le lecteur $drive n'existe pas." "ERROR"
    throw "Cible invalide."
  }
  $targetIsLocal = $true
} elseif ($BackupTarget -like "\\*") {
  if (-not (Test-Path $BackupTarget)) {
    Write-Log "Le partage $BackupTarget est injoignable." "ERROR"
    throw "Cible invalide."
  }
  Write-Log "ATTENTION: System State vers un partage réseau ne conserve qu'une seule version."
} else {
  Write-Log "Format de cible non reconnu: $BackupTarget" "ERROR"
  throw "Cible invalide."
}

# Lancer la sauvegarde
$vssArg = $UseVssCopy.IsPresent ? "-vssCopy" : "-vssFull"
$cmd = "wbadmin start systemstatebackup -backuptarget:`"$BackupTarget`" $vssArg -quiet"
Write-Log "Exécution: $cmd"
$proc = Start-Process -FilePath "cmd.exe" -ArgumentList "/c $cmd" -Wait -PassThru
if ($proc.ExitCode -ne 0) {
  Write-Log "Echec wbadmin (ExitCode=$($proc.ExitCode)). Voir les Journaux d'événements (Applications et Services Logs > Microsoft > Windows > Backup)." "ERROR"
  throw "wbadmin a échoué."
}
Write-Log "Sauvegarde System State terminée avec succès."

# Rotation (local seulement)
if ($targetIsLocal) {
  $del = "wbadmin delete systemstatebackup -keepVersions:$KeepVersions -backupTarget:`"$BackupTarget`" -quiet"
  Write-Log "Rotation des versions: $del"
  $proc2 = Start-Process -FilePath "cmd.exe" -ArgumentList "/c $del" -Wait -PassThru
  if ($proc2.ExitCode -eq 0) {
    Write-Log "Rotation OK (conservation $KeepVersions versions)."
  } else {
    Write-Log "Rotation: wbadmin a retourné ExitCode=$($proc2.ExitCode). Poursuite." "WARN"
  }
} else {
  Write-Log "Rotation ignorée (cible réseau : une seule version est conservée par Windows Server Backup)."
}

Write-Log "=== Fin sauvegarde System State ==="
exit 0
